function New-CustomEC2Instance {
	[CmdletBinding()]
	param
	(
		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$VpcCidrBlock,

		[Parameter()]
		[ValidateNotNullOrEmpty()]
		[switch]$EnableDnsSupport,
		
		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$SubnetCidrBlock,

		[Parameter(Mandatory)]
		[ValidateSet('Windows Server 2016')]
		[ValidateNotNullOrEmpty()]
		[string]$OperatingSystem,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$SubnetAvailabilityZone,

		[Parameter()]
		[ValidateNotNullOrEmpty()]
		[string]$InstanceType
	)

	## Tworzenie chmury VPC dla zakresu adresów CIDR zdefiniowanego przez użytkownika (o ile jeszcze nie istnieje)
	if (-not ($vpc = Get-EC2Vpc | where CidrBlock -eq $VpcCidrBlock)) {
		Write-Verbose -Message "Tworzenie nowej chmury VPC dla zakresu CIDR [$($VpcCidrBlock)]..."
		$vpc = New-EC2Vpc -CidrBlock $VpcCidrBlock
	} else {
		Write-Verbose -Message "Chmura VPC dla zakresu CIDR [$($VpcCidrBlock)] już istnieje."
	}
	
	## Włącz obsługę DNS dla chmury VPC (o ile wybrano taką opcję)
	if ($EnableDnsSupport.IsPresent) {
		Write-Verbose -Message "Włączanie obsługi DNS dla chmury VPC [$($vpc.VpcId)]..."
		## Dodaj obsługę DNS dla chmury VPC

		## Upewnij się, że wszystkie instancje EC2 dodane do tej chmury VPC poprawnie rozwiązują nazwy DNS,
		## Dzięki temu wszystkie instancje EC2 będą po utworzeniu wskazywać na serwery Amazon DNS 
		Edit-EC2VpcAttribute -VpcId $vpc.VpcId -EnableDnsSupport $true
		Edit-EC2VpcAttribute -VpcId $vpc.VpcId -EnableDnsHostnames $true
	}

	## Utwórz bramę internetową, która będzie działać jako router ruchu internetowego dla VPC (o ile taki router jeszcze nie istnieje)
	if (-not ($gw = Get-EC2InternetGateway | where { $_.Attachments.VpcId -eq $vpc.VpcId })) {
		Write-Verbose -Message 'Tworzenie nowej bramy internetowej...'
		$gw = New-EC2InternetGateway

		## Skojarz bramę internetową z VPC, aby instancje EC2 mogły kierować ruch do internetu
		Write-Verbose -Message "Przypisywanie bramy internetowej [$($gw.InternetGatewayID)] z chmurą VPC [$($vpc.VpcId)]..."
		Add-EC2InternetGateway -InternetGatewayId $gw.InternetGatewayId -VpcId $vpc.VpcId
	} else {
		Write-Verbose -Message "Brama internetowa została już przypisana do VPC [$($vpc.VpcId)]."
	}

	## Utwórz nową tablicę routingu dla VPC aby dodać domyślną trasę (o ile taka tablica jeszcze nie istnieje).
	if (-not ($rt = Get-EC2RouteTable -Filter @{ Name='vpc-id'; Value=$vpc.VpcId })) {
		Write-Verbose -Message "Tworzenie tablicy routingu dla VPC [$($vpc.VpcId)]..."
		$rt = New-EC2RouteTable -VpcId $vpc.VpcId
	} else {
		Write-Verbose -Message "Tablica routingu dla VPC [$($vpc.VpcId)] już istnieje."
	}
	
	## Utwórz domyślną trasę (o ile taka trasa jeszcze nie istnieje)
	if (-not ($rt.Routes | where {$_.DestinationCidrBlock -eq '0.0.0.0/0'})) {
		## Utwórz trasę domyślną (bramę domyślną) w tablicy routingu, wskazującą bramę internetową, 
		## aby zakończyć konfigurację i umożliwić instancjom EC2 podłączonym do VPC dostęp do internetu.
		Write-Verbose -Message "Tworzenie domyślnej trasy w tablicy routingu[$($rt.RouteTableId)]..."
		New-EC2Route -RouteTableId $rt.RouteTableId -GatewayId $gw.InternetGatewayId -DestinationCidrBlock '0.0.0.0/0'
	} else {
		Write-Verbose -Message "Domyślna trasa w tablicy routingu [$($rt.RouteTableId)] już istnieje."
	}

	## Utwórz podsieć i dodaj ją do tablicy routingu (o ile jeszcze taka podsieć nie istnieje)
	if (-not ($sn = Get-EC2Subnet -Filter @{ Name='vpc-id'; Value=$vpc.VpcId })) {
		Write-Verbose -Message "Tworzenie i rejestrowanie podsieci CIDR [$($SubnetCidrBlock)] dla chmury VPC [$($vpc.VpcId)]..."
		$sn = New-EC2Subnet -VpcId $vpc.VpcId -CidrBlock $SubnetCidrBlock -AvailabilityZone $SubnetAvailabilityZone
		Register-EC2RouteTable -RouteTableId $rt.RouteTableId -SubnetId $sn.SubnetId
	} else {
		Write-Verbose -Message "Podana podsieć już istnieje i została zarejestrowana dla chumury VPC [$($vpc.VpcId)]."
	}

	## Wyszukiwanie obrazu AMI
	switch ($OperatingSystem) {
		'Windows Server 2016' {
			$imageName = 'WINDOWS_2016_BASE'
		}
		default {
			throw "Nierozpoznany system operacyjny: [$_]"
		}
	}
	
	## Upewnij się, że obraz AMI został znaleziony; w przeciwnym wypadku zwróć błąd.
	if (-not ($ami = Get-EC2ImageByName -Name $imageName)) {
		Write-Error -Message "Nie znaleziono obrazu AMI o nazwie [$($imageName)]."
	} else {
		Write-Verbose -Message 'Tworzenie instancji EC2...'
		## Wybierz obraz AMI, którego będzie używać instancja; upewnij się, że instancja ma publiczne adresy IP;
		## określ typ instancji i dołącz utworzoną podsieć tak, aby karta sieciowa instancji mogła jej używać.
		$params = @{
			ImageId = $ami.ImageId
			AssociatePublicIp = $true
			InstanceType = $InstanceType
			SubnetId = $sn.SubnetId
		}
		New-EC2Instance @params
	}
}